#!/usr/bin/perl

#------------------------------------------------------------------------------
#
#   Description:
#
#       Cheap preprocesor emulator for Inform
#
#
#   Usage:
#
#       perl inf-tags.pl [inf-compiler] {inf-options} -L[language-tag] [src] {game}
#
#   or possibly:
#
#       ./inf-tags.pl [inf-compiler] {inf-options} -L[language-tag] [src] {game}
#
#   (Square brackets denote mandatory, curly braces optional items.)
#
#
#   Requirements:
#
#   This is a Perl script and requires Perl 5.
#
#
#   Synopsis:
#
#   This is a cheap preprocessor for a one-level conditional compilation of
#   Inform files. This could be used for a single source file containing
#   the data for different languages.
#
#   [inf-compiler] is the Inform compiler to use, perhaps with a full path
#   name prepended. [src] is the source file and {game} the game file to be
#   written which defaults the the source's base name with a z-Code suffix.
#
#   {inf-options} are the usual options of the Inform compiler. The language
#   tag must be specified. This can be any continous string of non-whitespace
#   characters like EN, EN-UK or ENGLISH. The tag is case-sensitive. There is
#   no default value.
#
#   Inside the source code, the text must be structured like this:
#
#       ...
#       description
#       :EN
#           "It is about two feet long.",
#       :ES
#           "Mide aproximadamente sesenta centmetros.",
#       :DE
#           "Es ist ungefhr einen halben Meter lang.",
#       :
#
#   That is, the text looks a bit like a switch statement. There is a line
#   for every tag, the whole block is terminated with a single colon. With the
#   -W option activated, a warning is issued when the block does not contain
#   a definition for the requested language as for example would be the case
#   if the above example hat be compiles with the tag IT. If the -W switch is
#   set the compilation will not be started if there are empty blocks for that
#   language tag.
#
#   The tagging follows Inform Include statements, but it skips system files.
#   This means that in any library file, the System_file; directive must be the
#   first non-comment.
#
#
#   Author/License:
#
#   Written by Martin Oehm in March 2006. Feel free to use the script if you
#   find it useful and to alter it according to your needs.
#
#   Write to <martin.oehm@gmx.de> for suggestions and bug reports.
#
#------------------------------------------------------------------------------

my @argv;                           # command line arguments

my $tag;                            # requested language tag
my $verbose;                        # be verbose on file inclusion
my $warn;                           # warning flag
my $nwarn = 0;                      # number of warnings

my $tmpline = 1;                    # current line in tmp file
my @tmpref;                         # reference line in tmp file
my @lineref;                        # reference line in original file
                                    # stored in the format "<number>:<filename>";
my $tmpbase;                        # base for temporary filenames
my $version = 5;                    # story file version, set by -v*
my $errorstyle = 0;                 # error message style as set by -E*
my $compiler = "";                  # compiler name
my $inputfile = "";                 # input file name, replaced with tmp
my $outputfile = "";                # output file, must be set explicitly
                                    # to avoid generic tmp-named story file
my @path;                           # search path for include files
my $cond = -1;                      # conditional flag:
                                    #   -1: true, outside tagged block
                                    #    0: false, inside tag
                                    #    1: true, inside tag
my $convert = 0;                    # true, if conversion is requested

sub resolve_icl {
    my $icl = shift;
    my $line;
    local *ICL;

    # TODO: make file handle local for recursive inclusion
    open ICL, "$1" or die "Could not open ICL file ($1).\n";

    while (defined ($line = <ICL>)) {
        chomp $line;

        #strip comments and whitespace
        $line =~ s/!.*//;
        $line =~ s/\s*$//;
        $line =~ s/^\s*//;

        my @a = split / +/, $line;
        for my $arg (@a) {
            if ($arg =~ /^[(](\S+)[)]/) {
                resolve_icl($1);
            } else {
                push @argv, $arg;
            }
        }
    }
}

# very basic and unsafe way to determine tmp basename
$tmpbase = sprintf "tmp_%x", rand(hex "100000");
$tmpbase = "a" if ($convert);

# push arguments onto a stack and resolve ICL files
while (defined (my $arg = shift)) {
    if ($arg =~ /^[(](\S+)[)]/) {
        resolve_icl($1);
    } else {
        push @argv, $arg;
    }
}

# strip out preprocessor directives and resolve basic flags
my $count = 0;
for my $arg (@argv) {
    if ($arg =~ /^-L(\S+)$/) {
        $tag = $1;
        delete @argv[$count];
    } elsif ($arg =~ /^\+include_path=(\S+)$/i) {
        push @path, split ",", $1;
    } elsif ($arg eq "-K") {
        $convert = 1;
    } elsif ($arg eq "-W") {
        $warn = 1;
        delete @argv[$count];
    } elsif ($arg eq "-V") {
        $verbose = 1;
        delete @argv[$count];
    } elsif ($arg =~ /^-/) {
        if ($arg =~ /E(\d)/) {
            $errorstyle = $1;
        } elsif ($arg =~ /v(\d)/) {
            $version = $1;
        } elsif ($arg =~ /G/) {
            $version = -1;
        }
    } elsif ($arg =~ /^[\+\$]/) {
        # leave $this and +that alone
    } else {
        if ($compiler eq "") {
            $compiler = $arg;
        } elsif ($inputfile eq "") {
            $inputfile = $arg;
            @argv[$count] = "$tmpbase.inf";
        } elsif ($outputfile eq "") {
            $outputfile = $arg;
        } else {
            die "Illegal extra data ($arg) at end of command.\n";
        }
    }
    $count++;
}

die "No input file was specified\n" if ($inputfile eq "");
die "No language tag was specified with -L\n" if ($tag eq "");

# determine output name if none was supplied
if ($outputfile eq "") {
    $outputfile = $inputfile;
    $outputfile =~ s/\.[^\.]+$//;
    if ($version < 0) {
        push @argv, "$outputfile.ulx";
        $def{"TARGET_GLULX"} = 0;
    } else {
        push @argv, "$outputfile.z$version";
        $def{"TARGET_ZCODE"} = 0;
    }
}

# open temporary input file
if (!$convert) {
    open TMP, ">$tmpbase.inf" or die "Could not open temporary file for writing.\n";
}

# write a line into the tmp file
sub echo {
    my $line = shift;
    return if ($cond == 0);
    if ($convert) {
        print $line;
    } else {
        print TMP $line;
    }
    $tmpline++;
}

# mark line reference
sub mark {
    my $m = shift;
    push @lineref, $m;
    push @tmpref, $tmpline;
}

# read and process an inputfile
sub process_file {
    my $infile = shift;
    my $usepath = shift;
    my $linenumber = 1;
    my $line;
    my $hit = 0;
    my %hitlist;
    local *IN;

    if (!(open IN, "<$infile" or open IN, "<$infile.inf" or open IN, "<$infile.h")) {
        if ($usepath) {
            my $found = 0;
            for my $p (@path) {
                if (open IN, "<$p/$infile"  or open IN, "<$p/file.inf"
                    or open IN, "<$p/$infile.h") {
                    $found = 1;
                    next;
                }
            }
            die "Could not find $infile in search path ",
                join(",", @path), ".\n" if (!$found);
        } else {
            die "Couldn't open input file $infile.\n";
        }
    }

    mark "$linenumber:$infile";

    while (defined ($line = <IN>)) {
        if ($line =~ s/^\s*://) {
            chomp $line;
            if ($line =~ s/^\s*(\S+)//) {
                # check tag
                if ($warn && exists($hitlist{$1})) {
                    print "$infile ($linenumber): Multiple definition of tag $1 in block.\n";
                    $nwarn++;
                }
                $hitlist{$1} = "";
                $cond = ($1 eq $tag);
                $hit |= $cond;
                echo("$line\n");
            } else {
                # empty tag: close block
                $cond = -1;
                if ($warn && $hit == 0) {
                    print "$infile ($linenumber): No section $tag in block.\n";
                    $nwarn++;
                }
                $hit = 0;
                %hitlist = ();
                echo("$line\n");
            }
            mark "$linenumber:$infile";
        } elsif ($cond != 0 && $line =~ /^\s*system_file/i) {
            # stop processing if the file is a system file
            print "$infile ($linenumber): Stop processing system file.\n" if ($verbose);
            if ($usepath) {
                echo("Include \"$infile\";\n");
            } else {
                echo("Include \">$infile\";\n");
            }
            return;
        } elsif ($cond != 0 && $line =~ /^\s*include\s*"([>]?)([^"]+)"/i) {
            # include files
            print "$infile ($linenumber): Including \"$1$2\".\n" if ($verbose);
            process_file($2, $1 eq "");
            mark "$linenumber:$infile";
        } else {
            echo($line);
        }
        $linenumber ++;
    }
}

process_file($inputfile, 0);
exit if $convert;                      # conversion runs stop here

mark(":");
close(TMP);

die "\nCompilation suppressed due to warnings.\n" if ($nwarn);

#
# run the Inform compiler with the temporary single source file and pass
# all compiler output to the temporary log.
#
# Note that on some systems the error output might be passed to stderr
# instead of stdout. In that case you might want to catch both streams
# using a csh >& syntax or some such.
#
system("@argv > $tmpbase.out");

# rearrange output
open TMP, "<$tmpbase.out" or die "Couldn't open temporary compiler output file\n";
$pos = 0;

while (defined(my $line = <TMP>)) {
    if ($errorstyle == 0 && $line =~ /^line (\d+):/) {
        # Acorn errors
        $nr = $1;
        while ($tmpref[$pos + 1] < $nr) { $pos++ };
        while ($tmpref[$pos] > $nr) { $pos-- };
        my @ref = split ":", $lineref[$pos];
        my $linenumber = $nr - $tmpref[$pos] + @ref[0] + 1;
        if (@ref[1] eq $inputfile) {
            $line =~ s/^line \d+:/line $linenumber:/;
        } else {
            $line =~ s/^line \d+:/"$ref[1]", line $linenumber:/;
        }
    } elsif ($errorstyle == 1 && $line =~ /^$tmpbase\.inf\s*[(](\d+)[)]:/) {
        # Microsoft errors
        $nr = $1;
        while ($tmpref[$pos + 1] < $nr) { $pos++ };
        while ($tmpref[$pos] > $nr) { $pos-- };
        my @ref = split ":", $lineref[$pos];
        my $linenumber = $nr - $tmpref[$pos] + @ref[0] + 1;
        $line =~ s/^$tmpbase\.inf[(](\d+)[)]:/$ref[1]($linenumber):/;
    } elsif ($errorstyle == 2 && $line =~ /^file\s+\"$tmpbase.inf\".\s+line\s+(\d+)/i) {
        # Mac style errors
        $nr = $1;
        while ($tmpref[$pos + 1] < $nr) { $pos++ };
        while ($tmpref[$pos] > $nr) { $pos-- };
        my @ref = split ":", $lineref[$pos];
        my $linenumber = $nr - $tmpref[$pos] + @ref[0] + 1;
        $line =~ s/^file\s+\"$tmpbase.inf\";\s+line\s+(\d+)/File "$ref[1]"; Line $linenumber/i;
    }
    print $line;
}


close (TMP);



# clean up temporary files after processing
unlink("$tmpbase.out") or die "I/O error on clean-up\n";
unlink("$tmpbase.inf") or die "I/O error on clean-up\n";